/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */

namespace org.owasp.appsensor.errors
{

    using System;
    using System.Text;
    using org.owasp.appsensor;
    using Owasp.Esapi;

    /**
     * This class is the security exception for AppSensor exceptions that don't occur in the 
     * context of a user. Rather than have an exception per detection point, this exception is used 
     * along with a member variable holding the detection point event code, 
     * ie. STE1, STE3, etc.
     * <p>
     * This allows applications to have a fairly simple mechanism to trigger
     * intrusion detection with appropriate handling for each event code.
     * (This assumes the AppSensorIntrusionDetector is used.)
     * <p>
     * To inform the IntrusionDetector that an AppSensor system exception has occurred, 
     * you simply need to add code similar to the below at the appropriate point in code: 
     * <code>
     * 	new AppSensorSystemException("STE3", "System Trend Exception", "High Number Of Same Transaction Across The Site.");
     * </code>
     * <p>
     * This code is nearly identical to the AppSensorException, however it is handled 
     * differently by the AppSensorIntrusionDetector
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     * @see org.owasp.esapi.errors.EnterpriseSecurityException
     */
    public class AppSensorSystemException : Exception
    {
        /** The Constant serialVersionUID. */
        private const long serialVersionUID = 1L;

        /** The logger. */
        protected ASLogger logger = APPSENSOR.AsUtilities.GetLogger("AppSensorSystemException");

        /** The detailed log message to be sent to the log file, not to be sent to the user */
        protected String logMessage = null;

        /** The unique event code for appsensor, eg RE1, ACE3, SE2, etc. */
        private String eventCode;

        /**
         * Instantiates a new intrusion trend exception specific to appsensor.
         * 
         * @param eventcode
         * 			the appsensor event code, ie. RE1, SE2, ACE5, etc.
         * @param userMessage
         * 			the message to display to users
         * @param logMessage
         * 			the message logged
         */
        public AppSensorSystemException(String eventCode, String userMessage, String logMessage)
            : base(userMessage)
        {
            this.logMessage = logMessage;
            this.eventCode = eventCode;
            logger.Error("AppSensorIntrusion - " + logMessage);
            Esapi.IntrusionDetector.AddException(this);
        }

        /**
         * Returns message meant for display to users
         *
         * Note that if you are unsure of what set this message, it would probably be a good idea to encode this message
         * before displaying it to the end user.
         * 
         * @return a String containing a message that is safe to display to users
         */
        public String UserMessage
        {
            get
            {
                return Message;
            }
        }

        /**
         * Returns a message that is safe to display in logs, but probably not to users
         * 
         * @return a String containing a message that is safe to display in logs, but probably not to users
         */
        public String LogMessage
        {
            get
            {
                return logMessage;
            }
        }

        /**
         * This method returns the AppSensor specific event code, such as RE1, SE2, ACE5, etc.
         * @return AppSensor specific event code, such as RE1, SE2, ACE5, etc.
         */
        public String EventCode
        {
            get
            {
                return this.eventCode;
            }
        }

        /**
         * Just an override of toString() for nice printing.
         */
        public override String ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.Append("Event code: ").Append(eventCode).Append(Environment.NewLine)
            .Append("User Message: ").Append(UserMessage).Append(Environment.NewLine)
            .Append("Log Message: ").Append(LogMessage).Append(Environment.NewLine);
            return sb.ToString();
        }

    }
}